-module(parallel_auth).

-export([request/3]).


request(Identity, Extra_, Options) ->
  %% events:info("Identity:~p Extra_:~p Options:~p", [Identity, Extra_, Options]),
  Extra = case Extra_ of undefined -> #{}; _ -> Extra_ end,
  {WhiteCountry, WhiteIP} = read_list(maps:get(whitelist, Extra, undefined)),
  %% events:info("WhiteIP:~p WhiteCountry:~p", [WhiteIP, WhiteCountry]),
  IP = proplists:get_value(ip, Identity),
  case lists:member(IP, WhiteIP) of
    true ->
      {ok, []};
    false ->
      {BlackCountry, BlackIP} = read_list(maps:get(blacklist, Extra, undefined)),
      %% events:info("BlackCountry:~p BlackIP:~p", [BlackCountry, BlackIP]),
      case lists:member(IP, BlackIP) of
        true ->
          {error, []};
        false ->
          Country = proplists:get_value(country, Options),
          case lists:member(Country, WhiteCountry) of
            true ->
              {ok, []};
            false ->
              case lists:member(Country, BlackCountry) of
                true ->
                  {error, []};
                false ->
                  verify_http(Identity, Extra, Options)
              end
          end
      end
  end.


read_list(undefined) ->
  {[], []};

read_list(Filename) ->
  Lines = read_lines(Filename),
  lists:partition(fun is_country/1, Lines).


is_country(<<C1,C2>>) 
  when C1 >= $A andalso C1 =< $Z andalso 
       ((C2 >= $A andalso C2 =< $Z) orelse (C2 >= $0 andalso C2 =< $0)) ->
  true;
is_country(<<"NONE">>) ->
  true;
is_country(_) ->
  false.

    

read_lines(Filename) ->
  case file:path_open(["priv", "/etc/flussonic"], Filename, [binary,read]) of
    {error, _} ->
      events:info("Cannot open file ~p", [Filename]),
      [];
    {ok, F, _Path} ->
      Read =
        fun Read() ->
            case file:read(F, 10240) of
              {ok, Bin} ->
                Data = Read(),
                <<Bin/binary, Data/binary>>;
              eof ->
                <<>>
            end
        end,
      Bin = Read(),
      file:close(F),
      Lines = [string:trim(Row) || Row <- binary:split(Bin,<<"\n">>, [global])],
      [Line || Line <- Lines, Line =/= <<>>]
  end.

              
verify_http(Identity, Extra, Options) ->
  Pid = self(),
  proc_lib:spawn(fun() ->
    Answer = parallel_request(Identity, Extra, Options),
    Pid ! {answer, Answer}
  end),
  receive
    {answer, Answer} -> Answer
  after
    15000 -> undefined
  end.


parallel_request(Identity, Extra, Options) ->
  Answer = case file:path_open(["priv", "/etc/flussonic"], "backends.txt", [binary,read]) of
    {error, _} ->
      {error, [{code,403},{message,<<"not configured backends">>}]};
    {ok, F, _Path} ->
      Bin = case file:pread(F, 0, 10240) of
        {ok, Bin_} -> Bin_;
        eof -> <<>>
      end,
      file:close(F),
      Backends = [Row || Row <- binary:split(Bin,<<"\n">>, [global]), size(Row) > 0],

      Self = self(),
      Processes = lists:map(fun(URL) ->
        Pid = proc_lib:spawn(fun() ->
          events:md([{media,proplists:get_value(name,Identity)}]),
          Reply = auth_http_backend:verify(URL, Extra, Identity, Options),
          Self ! {reply, self(), URL, Reply}
        end),
        erlang:monitor(process, Pid),
        Pid
      end, Backends),

      wait_for_reply(Processes)
  end,
  Answer.


wait_for_reply([]) ->
  {error, [{code,403}]};

wait_for_reply(Processes) ->
  receive
    {'DOWN', _, _, Pid, _} -> 
      wait_for_reply(lists:delete(Pid, Processes));
    {reply, Pid, _URL, {ok, Ok}} ->
      Processes1 = lists:delete(Pid, Processes),
      [erlang:exit(P, shutdown) || P <- Processes1],
      {ok, Ok};
    {reply, Pid, _URL, _} ->
      wait_for_reply(lists:delete(Pid, Processes))
  end.

