<?php
/*------------------------------------------------------------------------
  Solidres - Hotel booking extension for Joomla
  ------------------------------------------------------------------------
  @Author    Solidres Team
  @Website   http://www.solidres.com
  @Copyright Copyright (C) 2013 - 2017 Solidres. All Rights Reserved.
  @License   GNU General Public License version 3, or later
------------------------------------------------------------------------*/

defined('_JEXEC') or die;

/**
 * Utilities handler class
 * 
 * @package 	Solidres
 * @subpackage	Utilities
 */
class SRUtilities
{
	public static function translateDayWeekName($inputs)
	{
		$dayMapping = array(
			'0' => JText::_('sun'),
			'1' => JText::_('mon'),
			'2' => JText::_('tue'),
			'3' => JText::_('wed'),
			'4' => JText::_('thu'),
			'5' => JText::_('fri'),
			'6' => JText::_('sat')
		);

		foreach ($inputs as $input)
		{
			$input->w_day_name = $dayMapping[$input->w_day];
		}

		return $inputs;
	}

	public static function translateText($text)
	{
		if (strpos( $text, '{lang' ) !== false)
		{
			$text = self::filterText($text);
		}

		return $text;
	}

	public static function getTariffDetailsScaffoldings($config = array())
	{
		$scaffoldings = array();

		$dayMapping = array(
			'0' => JText::_('sun'),
			'1' => JText::_('mon'),
			'2' => JText::_('tue'),
			'3' => JText::_('wed'),
			'4' => JText::_('thu'),
			'5' => JText::_('fri'),
			'6' => JText::_('sat')
		);

		// If this is package per person or package per room
		if ($config['type'] == 2 || $config['type'] == 3 )
		{
			$scaffoldings[0] = new stdClass();
			$scaffoldings[0]->id = null;
			$scaffoldings[0]->tariff_id = $config['tariff_id'];
			$scaffoldings[0]->price = null;
			$scaffoldings[0]->w_day = 8;
			$scaffoldings[0]->guest_type = $config['guest_type'];
			$scaffoldings[0]->from_age = null;
			$scaffoldings[0]->to_age = null;
			$scaffoldings[0]->date = null;

			return $scaffoldings;
		}
		else // For normal complex tariff
		{
			if ($config['mode'] == 0) // Mode = Week
			{
				for ($i = 0; $i < 7; $i++)
				{
					$scaffoldings[$i] = new stdClass();
					$scaffoldings[$i]->id = null;
					$scaffoldings[$i]->tariff_id = $config['tariff_id'];
					$scaffoldings[$i]->price = null;
					$scaffoldings[$i]->w_day = $i;
					$scaffoldings[$i]->w_day_name = $dayMapping[$i];
					$scaffoldings[$i]->guest_type = $config['guest_type'];
					$scaffoldings[$i]->from_age = (strpos($config['guest_type'], 'child') !== false) ? 0 : null;
					$scaffoldings[$i]->to_age = (strpos($config['guest_type'], 'child') !== false) ? 10 : null;
					$scaffoldings[$i]->date = null;
				}

				return $scaffoldings;
			}
			else // Mode = Day
			{
				$tariffDates = self::calculateWeekDay($config['valid_from'], $config['valid_to']);
				$resultsSortedPerMonth = array();
				$resultsSortedPerMonth[date('Y-m', strtotime($tariffDates[0]))] = array();

				foreach ($tariffDates as $i => $tariffDate)
				{
					$scaffoldings[$i] = new stdClass();
					$scaffoldings[$i]->id = null;
					$scaffoldings[$i]->tariff_id = $config['tariff_id'];
					$scaffoldings[$i]->price = null;
					$scaffoldings[$i]->w_day = date('w', strtotime($tariffDate));
					$scaffoldings[$i]->w_day_name = $dayMapping[$scaffoldings[$i]->w_day];
					$scaffoldings[$i]->guest_type = $config['guest_type'];
					$scaffoldings[$i]->from_age = (strpos($config['guest_type'], 'child') !== false) ? 0 : null;
					$scaffoldings[$i]->to_age = (strpos($config['guest_type'], 'child') !== false) ? 10 : null;
					$scaffoldings[$i]->date = $tariffDate;

					$currentMonth = date('Y-m', strtotime($tariffDate));
					if (!isset($resultsSortedPerMonth[$currentMonth]))
					{
						$resultsSortedPerMonth[$currentMonth] = array();
					}

					$scaffoldings[$i]->w_day_label = $dayMapping[$scaffoldings[$i]->w_day] . ' ' . date('d', strtotime($scaffoldings[$i]->date));
					$scaffoldings[$i]->is_weekend = SRUtilities::isWeekend($scaffoldings[$i]->date);
					$scaffoldings[$i]->is_today = SRUtilities::isToday($scaffoldings[$i]->date);
					$resultsSortedPerMonth[$currentMonth][] = $scaffoldings[$i];
				}

				return $resultsSortedPerMonth;
			}
		}


	}

	/* Translate custom field by using language tag. Author: isApp.it Team */
	public static function getLagnCode()
	{
		$lang_codes = JLanguageHelper::getLanguages('lang_code');
		$lang_code 	= $lang_codes[JFactory::getLanguage()->getTag()]->sef;
		return $lang_code;
	}

	/* Translate custom field by using language tag. Author: isApp.it Team */
	public static function filterText($text)
	{
		if ( strpos( $text, '{lang' ) === false ) return $text;
		$lang_code = self::getLagnCode();
		$regex = "#{lang ".$lang_code."}(.*?){\/lang}#is";
		$text = preg_replace($regex,'$1', $text);
		$regex = "#{lang [^}]+}.*?{\/lang}#is";
		$text = preg_replace($regex,'', $text);
		return $text;
	}

	/**
	 * This simple function return a correct javascript date format pattern based on php date format pattern
	 *
	 **/
	public static function convertDateFormatPattern($input)
	{
		$mapping = array(
			'd-m-Y' => 'dd-mm-yy',
			'd/m/Y' => 'dd/mm/yy',
			'd M Y' => 'dd M yy',
			'd F Y' => 'dd MM yy',
			'D, d M Y' => 'D, dd M yy',
			'l, d F Y' => 'DD, dd MM yy',
			'Y-m-d' => 'yy-mm-dd',
			'm-d-Y' => 'mm-dd-yy',
			'm/d/Y' => 'mm/dd/yy',
			'M d, Y' => 'M dd, yy',
			'F d, Y' => 'MM dd, yy',
			'D, M d, Y' => 'D, M dd, yy',
			'l, F d, Y' => 'DD, MM dd, yy',
		);

		return $mapping[$input];
	}

	/**
	 * Get an array of week days in the period between $from and $to
	 *
	 * @param    string   From date
	 * @param    string   To date
	 *
	 * @return   array	  An array in format array(0 => 'Y-m-d', 1 => 'Y-m-d')
	 */
	public static function calculateWeekDay($from, $to)
	{
		$datetime1 	= new DateTime($from);
		$interval 	= self::calculateDateDiff($from, $to);
		$weekDays 	= array();

		$weekDays[] = $datetime1->format('Y-m-d');

		for ($i = 1; $i <= (int)$interval; $i++)
		{
			$weekDays[] = $datetime1->modify('+1 day')->format('Y-m-d');
		}

		return $weekDays;
	}

	/**
	 * Calculate the number of day from a given range
	 *
	 * Note: DateTime is PHP 5.3 only
	 *
	 * @param  string  $from   Begin of date range
	 * @param  string  $to     End of date range
	 * @param  string  $format The format indicator
	 *
	 * @return string
	 */
	public static function calculateDateDiff($from, $to, $format = '%a')
	{
		$datetime1 = new DateTime($from);
		$datetime2 = new DateTime($to);

		$interval = $datetime1->diff($datetime2);

		return $interval->format($format);
	}

	public static function getReservationStatusList($config = array())
	{
		// Build the active state filter options.
		$options = array();

		$options[] = JHtml::_('select.option', '0',  'SR_RESERVATION_STATE_PENDING_ARRIVAL');
		$options[] = JHtml::_('select.option', '1',  'SR_RESERVATION_STATE_CHECKED_IN');
		$options[] = JHtml::_('select.option', '2',  'SR_RESERVATION_STATE_CHECKED_OUT');
		$options[] = JHtml::_('select.option', '3',  'SR_RESERVATION_STATE_CLOSED');
		$options[] = JHtml::_('select.option', '4',  'SR_RESERVATION_STATE_CANCELED');
		$options[] = JHtml::_('select.option', '5',  'SR_RESERVATION_STATE_CONFIRMED');
		$options[] = JHtml::_('select.option', '-2', 'JTRASHED');
		$options[] = JHtml::_('select.option', '',   'JALL');

		return $options;
	}

	public static function getReservationPaymentStatusList($config = array())
	{
		// Build the active state filter options.
		$options = array();

		$options[] = JHtml::_('select.option', '0',  'SR_RESERVATION_PAYMENT_STATUS_UNPAID');
		$options[] = JHtml::_('select.option', '1',  'SR_RESERVATION_PAYMENT_STATUS_COMPLETED');
		$options[] = JHtml::_('select.option', '2',  'SR_RESERVATION_PAYMENT_STATUS_CANCELLED');
		$options[] = JHtml::_('select.option', '3',  'SR_RESERVATION_PAYMENT_STATUS_PENDING');
		$options[] = JHtml::_('select.option', '',   'SR_RESERVATION_PAYMENT_STATUS_ALL');

		return $options;
	}

	public static function removeArrayElementsExcept(&$array, $keyToRemain)
	{
		foreach ($array as $key => $val)
		{
			if ($key != $keyToRemain)
			{
				unset($array[$key]);
			}
		}
	}

	/**
	 * Check to see this user is asset's partner or not
	 *
	 * @param $joomlaUserId
	 * @param $assetId
	 *
	 * @return bool
	 *
	 */
	public static function isAssetPartner($joomlaUserId, $assetId)
	{
		JTable::addIncludePath(JPATH_ADMINISTRATOR.'/components/com_solidres/tables');
		JTable::addIncludePath(SRPlugin::getAdminPath('user') . '/tables');
		$tableAsset = JTable::getInstance('ReservationAsset', 'SolidresTable');
		$tableCustomer = JTable::getInstance('Customer', 'SolidresTable');
		$tableAsset->load($assetId);
		$tableCustomer->load(array('user_id' => $joomlaUserId));

		if ($tableAsset->partner_id == $tableCustomer->id)
		{
			return true;
		}

		return false;
	}

	/**
	 * Check to see if any of the given user groups is a Solidres partner group
	 *
	 * @param $joomlaUserGroups
	 *
	 * @return boolean
	 *
	 * @since 1.9.0
	 */
	public static function isPartnerGroups($joomlaUserGroups)
	{
		$solidresConfig = JComponentHelper::getParams('com_solidres');
		$partnerUserGroups = $solidresConfig->get('partner_user_groups', array());
		$partnerUserGroups = array_values($partnerUserGroups);

		if (count(array_intersect($partnerUserGroups, $joomlaUserGroups)) == 0)
		{
			return false;
		}

		return true;
	}

	/**
	 * Get the partner ID from the current logged in user
	 *
	 * @return bool
	 *
	 * @since 1.9.0
	 */
	public static function getPartnerId()
	{
		if (!SRPlugin::isEnabled('user'))
		{
			return false;
		}

		JTable::addIncludePath(SRPlugin::getAdminPath('user') . '/tables');
		$user = JFactory::getUser();

		if (!self::isPartnerGroups($user->getAuthorisedGroups()))
		{
			return false;
		}

		$customerTable = JTable::getInstance('Customer', 'SolidresTable');
		$customerTable->load(array('user_id' => $user->get('id')));

		return $customerTable->id;
	}

	public static function isApplicableForAdjoiningTariffs($roomTypeId, $checkIn, $checkOut, $excludes = array())
	{
		$result = array();

		$dbo = JFactory::getDbo();

		$query = '
				(SELECT DISTINCT t1.id
				FROM ' . $dbo->quoteName('#__sr_tariffs') . ' AS t1
				WHERE t1.valid_to >= '. $dbo->quote($checkIn). ' AND t1.valid_to <= '.$dbo->quote($checkOut).'
				AND t1.valid_from <= '.$dbo->quote($checkIn).' AND t1.state = 1 AND t1.room_type_id = ' . $roomTypeId . '
				'. (!empty($excludes) ? 'AND t1.id NOT IN (' . implode(',', $excludes) . ')' : '' ). '
				LIMIT 1)
				UNION ALL
				(SELECT DISTINCT t2.id
				FROM ' . $dbo->quoteName('#__sr_tariffs') . ' AS t2
				WHERE t2.valid_from <= '.$dbo->quote($checkOut).' AND t2.valid_from >= '.$dbo->quote($checkIn).'
				AND t2.valid_to >= '.$dbo->quote($checkOut).' AND t2.state = 1 AND t2.room_type_id = ' . $roomTypeId . '
				'. (!empty($excludes) ? 'AND t2.id NOT IN (' . implode(',', $excludes) . ')' : '' ). '
				LIMIT 1)
				';

		$tariffIds = $dbo->setQuery($query)->loadObjectList();

		if (count($tariffIds) == 2)
		{
			$query = 'SELECT datediff(t2.valid_from, t1.valid_to)
					FROM ' . $dbo->quoteName('#__sr_tariffs') . ' AS t1, ' . $dbo->quoteName('#__sr_tariffs') . ' AS t2
					WHERE t1.id = ' . $tariffIds[0]->id . ' AND t2.id = ' . $tariffIds[1]->id;

			if ($dbo->setQuery($query)->loadResult() == 1)
			{
				$result = array($tariffIds[0]->id, $tariffIds[1]->id);
			}
		}

		return $result;
	}

	public static function getUpdates()
	{
		$file = JPATH_ADMINISTRATOR . '/components/com_solidres/views/system/cache/updates.json';

		if (file_exists($file)
		    && ($raw = file_get_contents($file))
		    && ($updates = json_decode($raw, true))
		    && json_last_error() == JSON_ERROR_NONE
		)
		{
			return $updates;
		}

		return array();
	}

	/**
	 * Gets the update site Ids for our extension.
	 *
	 * @return 	mixed	An array of Ids or null if the query failed.
	 */
	public static function getUpdateSiteIds($extensionId)
	{
		$db = JFactory::getDbo();
		$query = $db->getQuery(true)
			->select($db->qn('update_site_id'))
			->from($db->qn('#__update_sites_extensions'))
			->where($db->qn('extension_id') . ' = ' . $db->q($extensionId));
		$db->setQuery($query);
		$updateSiteIds = $db->loadColumn(0);

		return $updateSiteIds;
	}

	public static function getExtension()
	{
		// Find the extension ID
		$db = JFactory::getDbo();
		$query = $db->getQuery(true)
			->select('*')
			->from($db->qn('#__extensions'))
			->where($db->qn('type') . ' = ' . $db->q('package'))
			->where($db->qn('element') . ' = ' . $db->q('pkg_solidres'));
		$db->setQuery($query);
		$extension = $db->loadObject();

		/*if (is_object($extension))
		{
			$this->extension_id = $extension->extension_id;
			$data = json_decode($extension->manifest_cache, true);

			if (isset($data['version']))
			{
				$this->version = $data['version'];
			}
		}*/

		return $extension;
	}

	public static function isWeekend($date)
	{
		return (date('N', strtotime($date)) >= 6);
	}

	public static function isToday($date)
	{
		return date('Y-m-d') == date('Y-m-d', strtotime($date));
	}

	public static function getCustomerGroupId()
	{
		JModelLegacy::addIncludePath(JPATH_COMPONENT_ADMINISTRATOR . '/models', 'SolidresModel');
		$user = JFactory::getUser();
		$customerGroupId = NULL;

		if (SR_PLUGIN_USER_ENABLED && $user->id > 0)
		{
			$customerModel = JModelLegacy::getInstance('Customer', 'SolidresModel', array('ignore_request' => true));
			$customer = $customerModel->getItem(array('user_id' => $user->id));
			$customerGroupId = ($customer) ? $customer->customer_group_id : NULL;
		}

		return $customerGroupId;
	}

	public static function getReservationStatus($status = null)
	{
		$statuses = array(
			0 => JText::_('SR_RESERVATION_STATE_PENDING_ARRIVAL'),
			1 => JText::_('SR_RESERVATION_STATE_CHECKED_IN'),
			2 => JText::_('SR_RESERVATION_STATE_CHECKED_OUT'),
			3 => JText::_('SR_RESERVATION_STATE_CLOSED'),
			4 => JText::_('SR_RESERVATION_STATE_CANCELED'),
			5 => JText::_('SR_RESERVATION_STATE_CONFIRMED'),
			-2 => JText::_('JTRASHED')
		);

		if (!empty($status))
		{
			return $statuses[$status];
		}

		return $statuses;
	}

	public static function areValidDatesForTariffLimit($checkin, $checkout, $tariffLimitCheckin, $tariffLimitCheckout='')
	{
		if (is_array($tariffLimitCheckin))
		{
			$limitCheckinArray = $tariffLimitCheckin;
		}
		else
		{
			$limitCheckinArray = json_decode($tariffLimitCheckin, true);
		}

		$checkinDate = new DateTime($checkin);
		$dayInfo = getdate($checkinDate->format('U'));

		// If the current check in date does not match the allowed check in dates, we ignore this tariff
		if (!in_array($dayInfo['wday'], $limitCheckinArray))
		{
			return false;
		}

		return true;
	}

	/**
	 * Check the given tariff to see if it satisfy the occupancy options
	 *
	 * @param   $complexTariff          The tariff to check for
	 * @param   $roomsOccupancyOptions  The selected occupancy options (could be for a single room or multi rooms)
	 * @return  boolean
	 *
	 * @since 2.2.0
	 */
	public static function areValidDatesForOccupancy($complexTariff, $roomsOccupancyOptions)
	{
		if (empty($roomsOccupancyOptions))
		{
			return true;
		}

		$isValidPeopleRange = true;
		$peopleRangeMatchCount = count($roomsOccupancyOptions);

		foreach ($roomsOccupancyOptions as $roomOccupancyOptions)
		{
			if (isset($roomOccupancyOptions['guests']))
			{
				$totalPeopleRequested = $roomOccupancyOptions['guests'];
			}
			else
			{
				$totalPeopleRequested = $roomOccupancyOptions['adults'] + $roomOccupancyOptions['children'];
			}

			if ($complexTariff->p_min > 0 && $complexTariff->p_max > 0)
			{
				$isValidPeopleRange = $totalPeopleRequested >= $complexTariff->p_min && $totalPeopleRequested <= $complexTariff->p_max;
			}
			elseif ( empty($complexTariff->p_min) && $complexTariff->p_max > 0)
			{
				$isValidPeopleRange = $totalPeopleRequested <= $complexTariff->p_max;
			}
			elseif ($complexTariff->p_min > 0 && empty($complexTariff->p_max))
			{
				$isValidPeopleRange = $totalPeopleRequested >= $complexTariff->p_min;
			}

			if (!$isValidPeopleRange)
			{
				$peopleRangeMatchCount--;
			}
		}

		if ($peopleRangeMatchCount == 0)
		{
			return false;
		}

		return true;
	}
}