<?php
/*------------------------------------------------------------------------
  Solidres - Hotel booking extension for Joomla
  ------------------------------------------------------------------------
  @Author    Solidres Team
  @Website   http://www.solidres.com
  @Copyright Copyright (C) 2013 - 2017 Solidres. All Rights Reserved.
  @License   GNU General Public License version 3, or later
------------------------------------------------------------------------*/

defined('_JEXEC') or die;

/**
 * Reservation handler class
 *
 * @package 	Solidres
 * @subpackage	Reservation
 */

class SRReservation
{
	/**
	 * The database object
	 *
	 * @var object
	 */
	protected $_dbo = null;

	public function __construct()
	{
		$this->_dbo = JFactory::getDbo();
	}

	/**
	 * Generate unique string for Reservation
	 *
	 * @param string $srcString The string that need to be calculate checksum
	 *
	 * @return string The unique string for each Reservation
	 */
	public function getCode($srcString)
	{
		return hash('crc32', (string) $srcString.uniqid());
	}

	/**
	 * Check a room to see if it is allowed to be booked in the period from $checkin -> $checkout
	 *
	 * @param int       $roomId
	 * @param string    $checkin
	 * @param string    $checkout
	 * @param int       $bookingType 0 is booking per night and 1 is booking per day
	 *
	 * @return boolean  True if the room is ready to be booked, False otherwise
	 */
	public function isRoomAvailable($roomId = 0, $checkin, $checkout, $bookingType = 0, $excludeId = 0)
	{
		$query = $this->_dbo->getQuery(true);
		$query->select('checkin, checkout');
		$query->from($this->_dbo->quoteName('#__sr_reservations').' as res');

		$query->join('INNER', $this->_dbo->quoteName('#__sr_reservation_room_xref').' as room
									ON res.id = room.reservation_id
									AND room.room_id = '.$this->_dbo->quote($roomId).($excludeId > 0 ? ' AND res.id != ' . (int) $excludeId : ''));
		$query->where('res.checkout >= '. $this->_dbo->quote(date('Y-m-d')));

		// By default, a room is occupied only with status = "Checked In" or "Confirmed"
		// Full list of reservation status
		// 0 : Pending arrival
		// 1 : Checked in
		// 2 : Checkout out
		// 3 : Closed
		// 4 : Cancelled
		// 5 : Confirmed
		$query->where('res.state = 1 or res.state = 5');

		$query->order('res.checkin');

		$this->_dbo->setQuery($query);
		// Get all current reservations and their checkin/checkout date
		$result = $this->_dbo->loadObjectList();

		if(is_array($result))
		{
			foreach($result as $currentReservation)
			{
				$currentCheckin = $currentReservation->checkin;
				$currentCheckout = $currentReservation->checkout;

				if ($checkin != $checkout) // Normal check
				{
					if (0 == $bookingType) // Per night
					{
						if
						(
							($checkin <= $currentCheckin && $checkout > $currentCheckin) ||
							($checkin >= $currentCheckin && $checkout <= $currentCheckout) ||
							($checkin < $currentCheckout && $checkout >= $currentCheckout)
						)
						{
							return false;
						}
					}
					else // Per day
					{
						if
						(
							($checkin <= $currentCheckin && $checkout >= $currentCheckin) ||
							($checkin >= $currentCheckin && $checkout <= $currentCheckout) ||
							($checkin <= $currentCheckout && $checkout >= $currentCheckout)
						)
						{
							return false;
						}
					}
				}
				else // Edge case when we check for available of a single date (checkin == checkout)
				{
					if (0 == $bookingType) // Per night
					{
						if
						(
							($checkin <= $currentCheckin && $checkout > $currentCheckin) ||
							($checkin >= $currentCheckin && $checkout < $currentCheckout) ||
							($checkin < $currentCheckout && $checkout >= $currentCheckout)
						)
						{
							return false;
						}
					}
					else
					{
						if
						(
							($checkin <= $currentCheckin && $checkout > $currentCheckin) ||
							($checkin >= $currentCheckin && $checkout < $currentCheckout) ||
							($checkin <= $currentCheckout && $checkout >= $currentCheckout)
						)
						{
							return false;
						}
					}
				}
			}
		}

		return true;
	}

	/**
	 * Check a room to see if it is allowed to be booked in the period from $checkin -> $checkout
	 *
	 * @param int    $result
	 * @param string $checkin
	 * @param string $checkout
	 * @param int    $bookingType 0 is booking per night and 1 is booking per day
	 *
	 * @return boolean  True if the room is ready to be booked, False otherwise
	 */
	public function isRoomLimited($result = 0, $checkin, $checkout, $bookingType = 0, $excludeId = 0)
	{
		$query = $this->_dbo->getQuery(true);

		$checkinMySQLFormat = "STR_TO_DATE(" . $this->_dbo->quote($checkin) . ", '%Y-%m-%d')";
		$checkoutMySQLFormat = "STR_TO_DATE(" . $this->_dbo->quote($checkout) . ", '%Y-%m-%d')";

		if (0 == $bookingType) // Booking per night
		{
			$query->select('COUNT(*) FROM '.$this->_dbo->quoteName('#__sr_limit_booking_details').'
									WHERE room_id = '.$this->_dbo->quote($result) . ' AND 
									limit_booking_id IN (SELECT id FROM '.$this->_dbo->quoteName('#__sr_limit_bookings').'
									WHERE
									(
										('.$checkinMySQLFormat.' <= start_date AND '.$checkoutMySQLFormat.' > start_date )
										OR
										('.$checkinMySQLFormat.' >= start_date AND '.$checkoutMySQLFormat.' <= end_date )
										OR
										(start_date != end_date AND '.$checkinMySQLFormat.' <= end_date AND '.$checkoutMySQLFormat.' >= end_date )
										OR
										(start_date = end_date AND '.$checkinMySQLFormat.' <= end_date AND '.$checkoutMySQLFormat.' > end_date )
									)
									AND state = 1
									)');
		}
		else // Booking per day
		{
			$query->select('COUNT(*) FROM '.$this->_dbo->quoteName('#__sr_limit_booking_details').'
									WHERE room_id = '.$this->_dbo->quote($result) . ' AND 
									limit_booking_id IN (SELECT id FROM '.$this->_dbo->quoteName('#__sr_limit_bookings').'
									WHERE
									(
										('.$checkinMySQLFormat.' <= start_date AND '.$checkoutMySQLFormat.' >= start_date )
										OR
										('.$checkinMySQLFormat.' >= start_date AND '.$checkoutMySQLFormat.' <= end_date )
										OR
										('.$checkinMySQLFormat.' <= end_date AND '.$checkoutMySQLFormat.' >= end_date )
									)
									AND state = 1
									)');
		}

		$this->_dbo->setQuery($query);
		$result = $this->_dbo->loadResult();

		if ( $result > 0)
		{
			return true;
		}

		return false;
	}

	/**
	 * Store reservation data and related data like children ages or other room preferences
	 *
	 * @param   int 	$reservationId
	 * @param   array 	$room Room information
	 *
	 * @return void
	 */
	public function storeRoom($reservationId, $room)
	{
		$query = $this->_dbo->getQuery(true);

		// Store main room info
		$query->insert($this->_dbo->quoteName('#__sr_reservation_room_xref'));
		$columns = array(
			$this->_dbo->quoteName('reservation_id'),
			$this->_dbo->quoteName('room_id'),
			$this->_dbo->quoteName('room_label'),
			$this->_dbo->quoteName('adults_number'),
			$this->_dbo->quoteName('children_number'),
			$this->_dbo->quoteName('guest_fullname'),
			$this->_dbo->quoteName('room_price'),
			$this->_dbo->quoteName('room_price_tax_incl'),
			$this->_dbo->quoteName('room_price_tax_excl')
		);

		if (isset($room['tariff_id']) && !is_null($room['tariff_id']))
		{
			$columns = array_merge($columns, array(
					$this->_dbo->quoteName('tariff_id'),
					$this->_dbo->quoteName('tariff_title'),
					$this->_dbo->quoteName('tariff_description')
				));
		}

		$query->columns($columns);

		$values = (int) $reservationId . ',' .
			$this->_dbo->quote($room['room_id']) . ',' .
			$this->_dbo->quote($room['room_label']) . ',' .
            $this->_dbo->quote(isset($room['adults_number']) ? $room['adults_number'] : 0)  . ',' .
            $this->_dbo->quote(isset($room['children_number']) ? $room['children_number'] : 0) . ',' .
            $this->_dbo->quote(isset($room['guest_fullname']) ? $room['guest_fullname'] : '') . ',' .
            $this->_dbo->quote(isset($room['room_price']) ? $room['room_price'] : 0) . ',' .
            $this->_dbo->quote(isset($room['room_price_tax_incl']) ? $room['room_price_tax_incl'] : 0) . ',' .
            $this->_dbo->quote(isset($room['room_price_tax_excl']) ? $room['room_price_tax_excl'] : 0);

		if (isset($room['tariff_id']) && !is_null($room['tariff_id']))
		{
			$values .= ',' .$this->_dbo->quote($room['tariff_id']) . ',' .
				$this->_dbo->quote($room['tariff_title']) . ',' .
				$this->_dbo->quote($room['tariff_description']);
		}

		$query->values($values);

		$this->_dbo->setQuery($query);
		$this->_dbo->execute();

		// Store related data
		$recentInsertedId = $this->_dbo->insertid();

		if (isset($room['children_number']))
		{
			for ($i = 0; $i < $room['children_number']; $i++)
			{
				$query->clear();
				$query->insert($this->_dbo->quoteName('#__sr_reservation_room_details'));
				$query->columns(array(
					$this->_dbo->quoteName('reservation_room_id'),
					$this->_dbo->quoteName('key'),
					$this->_dbo->quoteName('value')
				));
				$query->values(
					(int) $recentInsertedId . ',' .
					$this->_dbo->quote('child'.($i + 1)) . ',' .
					$this->_dbo->quote($room['children_ages'][$i])
				);

				$this->_dbo->setQuery($query);
				$this->_dbo->execute();
			}
		}

		if (isset($room['preferences']))
		{
			foreach ($room['preferences'] as $key => $value)
			{
				$query->clear();
				$query->insert($this->_dbo->quoteName('#__sr_reservation_room_details'));
				$query->columns(array(
					$this->_dbo->quoteName('reservation_room_id'),
					$this->_dbo->quoteName('key'),
					$this->_dbo->quoteName('value')
				));
				$query->values(
					(int) $recentInsertedId . ',' .
					$this->_dbo->quote($key) . ',' .
					$this->_dbo->quote($value)
				);

				$this->_dbo->setQuery($query);
				$this->_dbo->execute();
			}
		}
	}

	/**
	 * Store extra information
	 *
	 * @param  int      $reservationId
	 * @param  int      $roomId
	 * @param  string   $roomLabel
	 * @param  int      $extraId
	 * @param  string   $extraName
	 * @param  int      $extraQuantity The extra quantity or NULL if extra does not have quantity
	 * @param  int      $price
	 *
	 * @return void
	 */
	public function storeExtra($reservationId, $roomId, $roomLabel, $extraId, $extraName, $extraQuantity = NULL, $price = 0)
	{
		$query = $this->_dbo->getQuery(true);
		$query->insert($this->_dbo->quoteName('#__sr_reservation_room_extra_xref'));
		$query->columns(array(
			$this->_dbo->quoteName('reservation_id'),
			$this->_dbo->quoteName('room_id'),
			$this->_dbo->quoteName('room_label'),
			$this->_dbo->quoteName('extra_id'),
			$this->_dbo->quoteName('extra_name'),
			$this->_dbo->quoteName('extra_quantity'),
			$this->_dbo->quoteName('extra_price')
		));
		$query->values(
			$this->_dbo->quote($reservationId) . ',' .
			$this->_dbo->quote($roomId) . ',' .
			$this->_dbo->quote($roomLabel) . ',' .
			$this->_dbo->quote($extraId) . ',' .
			$this->_dbo->quote($extraName) . ',' .
			($extraQuantity === NULL ? NULL : $this->_dbo->quote($extraQuantity) ) . ',' .
			$this->_dbo->quote($price)
		);
		$this->_dbo->setQuery($query);
		$this->_dbo->execute();
	}

	/**
	 * Check for the validity of check in and check out date
	 *
	 * Conditions
	 * + Number of days a booking must be made in advance
	 * + Maximum length of stay
	 *
	 * @param string $checkIn
	 * @param string $checkOut
	 * @param array $conditions
	 *
	 * @throws Exception
	 * @return Boolean
	 */
	public function isCheckInCheckOutValid($checkIn, $checkOut, $conditions)
	{
		$checkIn = new DateTime($checkIn);
		$checkOut = new DateTime($checkOut);
		$today = new DateTime(date('Y-m-d'));

		if ($checkIn < $today || $checkOut < $today)
		{
			throw new Exception('SR_ERROR_PAST_CHECKIN_CHECKOUT', 50005);
		}

		if (!isset($conditions['booking_type']) || (isset($conditions['booking_type']) && $conditions['booking_type'] == 0))
		{
			if ($checkOut <= $checkIn)
			{
				throw new Exception('SR_ERROR_INVALID_CHECKIN_CHECKOUT', 50001);
			}
		}

		// Interval between check in and check out date
		if (isset($conditions['min_length_of_stay']) && $conditions['min_length_of_stay'] > 0)
		{
			$interval1 = $checkOut->diff($checkIn)->format('%a');

			if ( $interval1 < $conditions['min_length_of_stay']) // count nights, not days
			{
				throw new Exception('SR_ERROR_INVALID_MIN_LENGTH_OF_STAY_' . $conditions['booking_type'], 50002);
			}
		}

		// Interval between checkin and today
		$interval2 = $checkIn->diff($today)->format('%a');

		if (isset($conditions['min_days_book_in_advance']) && $conditions['min_days_book_in_advance'] > 0)
		{
			if ($interval2 < $conditions['min_days_book_in_advance'])
			{
				throw new Exception('SR_ERROR_INVALID_MIN_DAYS_BOOK_IN_ADVANCE', 50003);
			}
		}

		if (isset($conditions['max_days_book_in_advance']) && $conditions['max_days_book_in_advance'] > 0)
		{
			if ($interval2 > $conditions['max_days_book_in_advance'])
			{
				throw new Exception('SR_ERROR_INVALID_MAX_DAYS_BOOK_IN_ADVANCE', 50004);
			}
		}

		return true;
	}

	/**
	 * Send email when reservation is completed
	 *
	 * @since  0.1.0
	 *
	 * @param int       $reservationId    The reservation to get the reservation info for emails (Optional)
	 * @param string    $action           The purpose of this email, default is for new reservation notification
	 *
	 * @return boolean True if email sending completed successfully. False otherwise
	 */
	public function sendEmail($reservationId = null, $action = '')
	{
		$lang = JFactory::getLanguage();
		$lang->load('com_solidres', JPATH_ADMINISTRATOR  . '/components/com_solidres', null, 1);
		$lang->load('com_solidres', JPATH_SITE . '/components/com_solidres');
		$subject = array();
		$body = array();
		$emailFormat = 'text/html';
		$solidresConfig = JComponentHelper::getParams('com_solidres');
		$dateFormat = $solidresConfig->get('date_format', 'd-m-Y');
		$config = JFactory::getConfig();
		$app = JFactory::getApplication();
		$context = 'com_solidres.reservation.process';
		$tzoffset = $config->get('offset');
		$timezone = new DateTimeZone($tzoffset);
		$messageTemplateExt = ($emailFormat == 'text/html' ? 'html' : 'txt') ;
		if (isset($reservationId))
		{
			$savedReservationId = $reservationId;
		}
		else
		{
			$savedReservationId = $app->getUserState($context.'.savedReservationId');
		}

		JModelLegacy::addIncludePath(JPATH_ADMINISTRATOR . '/components/com_solidres/models', 'SolidresModel');
		JTable::addIncludePath(JPATH_ADMINISTRATOR . '/components/com_solidres/tables', 'SolidresTable');
		$modelReservation = JModelLegacy::getInstance('Reservation', 'SolidresModel', array('ignore_request' => true));
		$savedReservationData = $modelReservation->getItem($savedReservationId);
		JLoader::register('SRUtilities', SRPATH_LIBRARY . '/utilities/utilities.php');
		$stayLength = (int)SRUtilities::calculateDateDiff($savedReservationData->checkin, $savedReservationData->checkout);
		$direction = JFactory::getDocument()->direction;

		$modelAsset = JModelLegacy::getInstance('ReservationAsset', 'SolidresModel', array('ignore_request' => true));
		$asset = $modelAsset->getItem($savedReservationData->reservation_asset_id);

		// Load override language file
		$lang->load('com_solidres_category_'.$asset->category_id, JPATH_COMPONENT, null, true);

		$hotelEmail = $asset->email;
		$hotelName = $asset->name;
		$customerEmail = $savedReservationData->customer_email;
		$hotelEmailList[] = $hotelEmail;
		$enableTouristTax = false;
		if (isset($asset->params['enable_tourist_tax']))
		{
			$enableTouristTax = $asset->params['enable_tourist_tax'];
		}

		switch ($action)
		{
			case 'reservation_status_pending_to_cancelled':
			case 'reservation_status_checkedin_to_cancelled':
			case 'reservation_status_checkedout_to_cancelled':
			case 'reservation_status_closed_to_cancelled':
			case 'reservation_status_confirmed_to_cancelled':
				$subject[$customerEmail] = JText::_('SR_EMAIL_RESERVATION_CANCELLED');
				$subject[$hotelEmail] = JText::sprintf(
					'SR_EMAIL_RESERVATION_CANCELLED_NOTIFICATION',
					$savedReservationData->code,
					$savedReservationData->customer_firstname,
					$savedReservationData->customer_lastname
				);

				break;
			case ''; // When new reservation is created for the first time
				$subject[$customerEmail] = JText::_('SR_EMAIL_RESERVATION_COMPLETE');
				$subject[$hotelEmail] = JText::sprintf(
					'SR_EMAIL_NEW_RESERVATION_NOTIFICATION',
					$savedReservationData->code,
					$savedReservationData->customer_firstname,
					$savedReservationData->customer_lastname
				);
				break;
			default:
				return true; // Default: do not send any emails
				break;
		}

		// If User plugin is installed and enabled
		if (SRPlugin::isEnabled('user') && SRPlugin::isEnabled('hub') && !is_null($asset->partner_id))
		{
			JModelLegacy::addIncludePath(SRPlugin::getAdminPath('user') . '/models', 'SolidresModel');
			$modelCustomer = JModelLegacy::getInstance('Customer', 'SolidresModel', array('ignore_request' => true));
			$partner = $modelCustomer->getItem($asset->partner_id);
			if (!empty($partner->email) && $partner->email != $hotelEmail )
			{
				$hotelEmailList[] = $partner->email;
				$subject[$partner->email] = $subject[$hotelEmail];
			}
		}

		$bankWireInstructions = array();

		if ($savedReservationData->payment_method_id == 'bankwire')
		{
			$solidresPaymentConfigData = new SRConfig(array('scope_id' => $savedReservationData->reservation_asset_id));
			$bankWireInstructions['account_name'] = SRUtilities::translateText($solidresPaymentConfigData->get('payments/bankwire/bankwire_accountname'));
			$bankWireInstructions['account_details'] = SRUtilities::translateText($solidresPaymentConfigData->get('payments/bankwire/bankwire_accountdetails'));
		}

		// We are free to choose between the inliner version and noninliner version
		// Inliner version is hard to maintain but it displays well in gmail (web).
		if ($app->isAdmin())
		{
			$reservationCompleteCustomerEmailTemplate = new JLayoutFile(
				'emails.reservation_complete_customer_' . $messageTemplateExt . '_inliner',
				JPATH_ROOT . '/components/com_solidres/layouts'
			);
		}
		else
		{
			$reservationCompleteCustomerEmailTemplate = new JLayoutFile('emails.reservation_complete_customer_'.$messageTemplateExt.'_inliner');
		}

		if ($app->isAdmin())
		{
			$reservationCompleteOwnerEmailTemplate = new JLayoutFile(
				'emails.reservation_complete_owner_html_inliner',
				JPATH_ROOT . '/components/com_solidres/layouts'
			);
		}
		else
		{
			$reservationCompleteOwnerEmailTemplate = new JLayoutFile('emails.reservation_complete_owner_html_inliner');
		}

		// Prepare some currency data to be showed
		$baseCurrency = new SRCurrency(0, $savedReservationData->currency_id);
		$subTotal = clone $baseCurrency;
		$subTotal->setValue($savedReservationData->total_price_tax_excl);

		$discountTotal = clone $baseCurrency;
		$discountTotal->setValue($savedReservationData->total_discount);

		$tax = clone $baseCurrency;
		$tax->setValue($savedReservationData->tax_amount);
		$touristTax = clone $baseCurrency;
		$touristTax->setValue($savedReservationData->tourist_tax_amount);

		$paymentMethodSurcharge = clone $baseCurrency;
		$paymentMethodSurcharge->setValue($savedReservationData->payment_method_surcharge);
		$paymentMethodDiscount = clone $baseCurrency;
		$paymentMethodDiscount->setValue($savedReservationData->payment_method_discount);

		$totalExtraPriceTaxExcl = clone $baseCurrency;
		$totalExtraPriceTaxExcl->setValue($savedReservationData->total_extra_price_tax_excl);
		$extraTax = clone $baseCurrency;
		$extraTax->setValue($savedReservationData->total_extra_price_tax_incl - $savedReservationData->total_extra_price_tax_excl);

		$grandTotal = clone $baseCurrency;
		if ($savedReservationData->discount_pre_tax)
		{
			$grandTotalAmount = $savedReservationData->total_price_tax_excl - $savedReservationData->total_discount + $savedReservationData->tax_amount + $savedReservationData->total_extra_price;
		}
		else
		{
			$grandTotalAmount = $savedReservationData->total_price_tax_excl + $savedReservationData->tax_amount  - $savedReservationData->total_discount + $savedReservationData->total_extra_price;
		}
		$grandTotalAmount += isset($savedReservationData->tourist_tax_amount) ? $savedReservationData->tourist_tax_amount : 0;
		$grandTotalAmount += isset($savedReservationData->payment_method_surcharge) ? $savedReservationData->payment_method_surcharge : 0;
		$grandTotalAmount -= isset($savedReservationData->payment_method_discount) ? $savedReservationData->payment_method_discount : 0;
		$grandTotal->setValue($grandTotalAmount);

		$depositAmount = clone $baseCurrency;
		$depositAmount->setValue(isset($savedReservationData->deposit_amount) ? $savedReservationData->deposit_amount : 0);

		// Get custom field if available
		$reservationCustomerField = null;
		if (SRPlugin::isEnabled('customfield'))
		{
			$customFieldContext = 'com_solidres.customer.' . $savedReservationData->id;
			$reservationCustomFields = SRCustomFieldHelper::getValues(array('context' => $customFieldContext));
			require_once SRPlugin::getAdminPath('customfield') . '/helpers/customfieldvalue.php';
			$reservationCustomerField = new SRCustomFieldValue($reservationCustomFields);
		}

		$displayData = array(
			'reservation' => $savedReservationData,
			'reservation_custom_field' => $reservationCustomerField,
			'sub_total' => $subTotal->format(),
			'total_discount' => $savedReservationData->total_discount > 0.00 ? $discountTotal->format() : NULL,
			'tax' => $tax->format(),
			'tourist_tax' => $touristTax->format(),
			'total_extra_price_tax_excl' => $totalExtraPriceTaxExcl->format(),
			'extra_tax' => $extraTax->format(),
			'grand_total' => $grandTotal->format(),
			'stay_length' => $stayLength,
			'deposit_amount' => $depositAmount->format(),
			'bankwire_instructions' => $bankWireInstructions,
			'asset' => $asset,
			'date_format' => $dateFormat,
			'timezone' => $timezone,
			'base_currency' => $baseCurrency,
			'payment_method_custom_email_content' => $app->getUserState( $context . '.payment_method_custom_email_content', ''),
			'discount_pre_tax' => $savedReservationData->discount_pre_tax,
			'direction' => $direction,
			'enable_tourist_tax' => $enableTouristTax,
			'payment_method_surcharge' => $paymentMethodSurcharge->format(),
			'payment_method_discount' => $paymentMethodDiscount->format()
		);

		switch ($action)
		{
			case 'reservation_status_pending_to_cancelled':
			case 'reservation_status_checkedin_to_cancelled':
			case 'reservation_status_checkedout_to_cancelled':
			case 'reservation_status_closed_to_cancelled':
			case 'reservation_status_confirmed_to_cancelled':
				$displayData['greetingText'] = JText::sprintf('SR_EMAIL_GREETING_TEXT_CANCELLED', $savedReservationData->code, $displayData['asset']->name);
				break;
			default:
				$displayData['greetingText'] = JText::sprintf('SR_EMAIL_GREETING_TEXT', $displayData['asset']->name);
				break;
		}

		$body[$customerEmail] = $reservationCompleteCustomerEmailTemplate->render($displayData);

		// Send email to customer
		$mail = SRFactory::get('solidres.mail.mail');
		$mail->setSender(array($hotelEmail, $hotelName));
		$mail->addRecipient($customerEmail);
		$mail->setSubject($subject[$customerEmail]);
		$mail->setBody($body[$customerEmail]);

		if (SRPlugin::isEnabled('invoice'))
		{
			// This is a workaround for this Joomla's bug  https://github.com/joomla/joomla-cms/issues/3451
			// When it is fixed, update this logic
			if (file_exists(JPATH_BASE . '/templates/' . $app->getTemplate() . '/html/layouts/com_solidres/emails/reservation_complete_customer_pdf.php' ))
			{
				$reservationCompleteCustomerPdfTemplate = new JLayoutFile('emails.reservation_complete_customer_pdf');
			}
			else
			{
				$reservationCompleteCustomerPdfTemplate = new JLayoutFile(
					'emails.reservation_complete_customer_pdf',
					JPATH_ROOT . '/plugins/solidres/invoice/layouts'
				);
			}

			$pdf = NULL;
			$pdf = $reservationCompleteCustomerPdfTemplate->render($displayData);

			if($solidresConfig->get('enable_pdf_attachment',1) == 1)
			{
				$this->getPDFAttachment($mail, $pdf, $savedReservationId, $savedReservationData->code);
			}

			$selectedPaymentMethod = $app->getUserState($context.'.payment_method_id', '');
			$autoSendPaymentMethods = $solidresConfig->get('auto_send_invoice_payment_methods','');
			if($solidresConfig->get('auto_send_invoice',0) == 1
			   && in_array($selectedPaymentMethod, $autoSendPaymentMethods)
			)
			{
				JPluginHelper::importPlugin('solidres');
				$invoiceFolder  = JPATH_ROOT . '/media/com_solidres/invoices/';
				$invoiceTable = JTable::getInstance('Invoice', 'SolidresTable');
				$invoiceTable->load(array('reservation_id' => $savedReservationId));
				if ($invoiceTable->id)
				{
					$mail->addAttachment($invoiceFolder . $invoiceTable->filename, 'Invoice_' . $invoiceTable->invoice_number . '.pdf', 'base64', 'application/pdf' );
				}
			}
		}

		$mail->IsHTML($emailFormat == 'text/html' ? true : false);

		if (!$mail->send())
		{
			return false;
		}

		// Send to the hotel owner
		$editLinks = array(
			'admin'     => JUri::root() . 'administrator/index.php?option=com_solidres&view=reservation&layout=edit&id='.$displayData['reservation']->id,
			'partner'   => JUri::root() . 'index.php?option=com_solidres&task=reservationform.edit&id='.$displayData['reservation']->id
		);

		$displayData['editLink'] = $editLinks['admin'];
		switch ($action)
		{
			case 'reservation_status_pending_to_cancelled':
			case 'reservation_status_checkedin_to_cancelled':
			case 'reservation_status_checkedout_to_cancelled':
			case 'reservation_status_closed_to_cancelled':
			case 'reservation_status_confirmed_to_cancelled':
				$displayData['greetingText'] = JText::sprintf('SR_EMAIL_NOTIFICATION_GREETING_TEXT_CANCELLED', $displayData['reservation']->code, $editLinks['admin']);
				break;
			default:
				$displayData['greetingText'] = JText::sprintf('SR_EMAIL_NOTIFICATION_GREETING_TEXT', $editLinks['admin']);
				break;
		}
		$body[$hotelEmail] = $reservationCompleteOwnerEmailTemplate->render($displayData);

		$mail2 = SRFactory::get('solidres.mail.mail');
		$mail2->setSender(array($hotelEmail, $hotelName));
		$mail2->addRecipient($hotelEmail);
		$additionalNotificationEmails = isset($asset->params['additional_notification_emails']) ? explode(',', $asset->params['additional_notification_emails']) : array();
		if (!empty($additionalNotificationEmails))
		{
			$mail2->addRecipient($additionalNotificationEmails);
		}
		$mail2->setSubject($subject[$hotelEmail]);
		$mail2->setBody($body[$hotelEmail]);
		$mail2->IsHTML($emailFormat == 'text/html' ? true : false);

		if (!$mail2->send())
		{
			return false;
		}

		// Send to the hotel partner
		if (SRPlugin::isEnabled('user') && SRPlugin::isEnabled('hub') && isset($partner->email))
		{
			switch ($action)
			{
				case 'reservation_status_pending_to_cancelled':
				case 'reservation_status_checkedin_to_cancelled':
				case 'reservation_status_checkedout_to_cancelled':
				case 'reservation_status_closed_to_cancelled':
				case 'reservation_status_confirmed_to_cancelled':
					$displayData['greetingText'] = JText::sprintf('SR_EMAIL_NOTIFICATION_GREETING_TEXT_CANCELLED', $displayData['reservation']->code, $editLinks['partner']);
					break;
				default:
					$displayData['greetingText'] = JText::sprintf('SR_EMAIL_NOTIFICATION_GREETING_TEXT', $editLinks['partner']);
					break;
			}

			$displayData['editLink'] = $editLinks['partner'];
			$body[$partner->email] = $reservationCompleteOwnerEmailTemplate->render($displayData);

			$mail3 = SRFactory::get('solidres.mail.mail');
			$mail3->setSender(array($hotelEmail, $hotelName));
			$mail3->addRecipient($partner->email);
			$mail3->setSubject($subject[$partner->email]);
			$mail3->setBody($body[$partner->email]);
			$mail3->IsHTML($emailFormat == 'text/html' ? true : false);

			if (!$mail3->send())
			{
				return false;
			}
		}

		return true;
	}

	/**
	 * Create PDF attachment.
	 * @param $mail		mail object.
	 * @param $reid		reservation id.
	 * @param $reCode	reservation code.
	 */
	protected function getPDFAttachment($mail, $content, $reid, $reCode)
	{
		$dispatcher = JEventDispatcher::getInstance();
		JPluginHelper::importPlugin('solidres');
		$solidresConfig = JComponentHelper::getParams('com_solidres');
		$attachmentFileName = $solidresConfig->get('solidres_invoice_pdf_file_name', 'voucher');
		$results = $dispatcher->trigger('onSolidresReservationEmail', array($content, $reid));

		if($results)
		{
			$mail->addAttachment($results[0], $attachmentFileName . '_'.$reCode.'.pdf', 'base64', 'application/pdf' );
		}
	}
}