<?php
/**
 * Mobile Joomla! extension
 * https://www.mobilejoomla.com
 *
 * @version    2.1.51
 * @license    GNU/GPL v2 - http://www.gnu.org/licenses/gpl-2.0.html
 * @copyright  (C) 2008-2022 Kuneri Ltd. / Denis Ryabov
 * @date       March 2022
 */
defined('_JEXEC') or die('Restricted access');

class MjInspection
{
    /** @var array */
    private $blob;

    /** @var bool */
    private $isJoomla15;

    /**
     * @param MjSettingsModel $mjSettings
     * @return array
     */
    public function getWarnings($mjSettings)
    {
        $this->blob = array();
        $this->isJoomla15 = (strncmp(JVERSION, '1.5.', 4) === 0);
        $this->checkEnabled();
        $this->checkLegacy();
        $this->checkGD2();
        $this->checkRemoteConnection();
        $this->checkAliasDuplicates();
        $this->checkTemplateAssignments();
        $this->checkForcedMarkup();
        $this->checkFavicon($mjSettings);
        $this->checkPluginsOrdering();
        return $this->blob;
    }

    private function checkEnabled()
    {
        $mjEnabled = JPluginHelper::isEnabled('system', 'mobilejoomla');
        $mjOptEnabled = JPluginHelper::isEnabled('system', 'mobilejoomlaopt');
        if ($mjEnabled && $mjOptEnabled) {
            return;
        }

        $joomlaWrapper = MjJoomlaWrapper::getInstance();
        $db = $joomlaWrapper->getDbo();
        $query = new MjQueryBuilder($db);
        if ($this->isJoomla15) {
            $pluginUrl = 'index.php?option=com_plugins&view=plugin&client=site&task=edit&cid[]=';
            $rows = $query
                ->select('id', 'element')
                ->from('#__plugins')
                ->where($query->qn('element') . ' IN (' . $query->q('mobilejoomla') . ', ' . $query->q('mobilejoomlaopt') . ')')
                ->setQuery()
                ->loadObjectList('element');
        } else {
            $pluginUrl = 'index.php?option=com_plugins&task=plugin.edit&extension_id=';
            $rows = $query
                ->select('element', $query->qn('extension_id') . ' AS ' . $query->qn('id'))
                ->from('#__extensions')
                ->where($query->qn('type') . '=' . $query->q('plugin'))
                ->where($query->qn('element') . ' IN (' . $query->q('mobilejoomla') . ', ' . $query->q('mobilejoomlaopt') . ')')
                ->setQuery()
                ->loadObjectList('element');
        }

        $input = array();
        if (!$mjEnabled) {
            $title = 'System - Mobile Core';
            if (isset($rows['mobilejoomla'])) {
                $title = '<a href="' . $pluginUrl . $rows['mobilejoomla']->id . '">' . $title . '</a>';
            }
            $input[] = $title;
        }
        if (!$mjOptEnabled) {
            $title = 'System - Mobile Page Optimizer';
            if (isset($rows['mobilejoomlaopt'])) {
                $title = '<a href="' . $pluginUrl . $rows['mobilejoomlaopt']->id . '">' . $title . '</a>';
            }
            $input[] = $title;
        }
        $this->blob[] = array(
            'label' => MjHtml::label('', 'COM_MJ__WARNING_DISABLED_PLUGINS'),
            'input' => MjHtml::text(implode('<br>', $input))
        );
    }

    private function checkLegacy()
    {
        $legacyPlugins = array(
            'system/mobilebot' => 'COM_MJ__WARNING_LEGACY_PLUGIN_MOBILEBOT',
            'mobile/adremover' => 'COM_MJ__WARNING_LEGACY_PLUGIN_ADREMOVER',
            'mobile/mjtune' => 'COM_MJ__WARNING_LEGACY_PLUGIN_MJTUNE',
            'mobile/scientia' => 'COM_MJ__WARNING_LEGACY_PLUGIN_SCIENTIA',
            'mobile/terawurfl' => 'COM_MJ__WARNING_LEGACY_PLUGIN_TERAWURFL',
        );
        $warnings = array();
        foreach ($legacyPlugins as $plugin => $help) {
            list($type, $name) = explode('/', $plugin);
            if (JPluginHelper::isEnabled($type, $name)) {
                $warnings[] = '<b>' . $plugin . '</b>: ' . JText::_($help);
            }
        }
        if (count($warnings)) {
            $this->blob[] = array(
                'label' => MjHtml::label('', 'COM_MJ__WARNING_LEGACY_EXTENSIONS'),
                'input' => MjHtml::text(implode('<br>', $warnings))
            );
        }
    }

    private function checkGD2()
    {
        if (!function_exists('imagecopyresized')) {
            $this->blob[] = array(
                'label' => MjHtml::label('', 'COM_MJ__WARNING_GD2'),
                'input' => MjHtml::text(JText::_('COM_MJ__WARNING_GD2_TEXT'))
            );
        }
    }

    private function checkRemoteConnection()
    {
        if (!preg_match('#\.pro$#', '2.1.51')) {
            return;
        }
        if (!function_exists('fsockopen')
            && !function_exists('curl_init')
            && !ini_get('allow_url_fopen')
        ) {
            $this->blob[] = array(
                'label' => MjHtml::label('', 'COM_MJ__WARNING_REMOTE'),
                'input' => MjHtml::text(JText::_('COM_MJ__WARNING_REMOTE_TEXT'))
            );
        }
    }

    private function checkAliasDuplicates()
    {
        if (!$this->isJoomla15) {
            return;
        }

        $db = JFactory::getDbo();
        $query = "SELECT m1.id, m1.menutype, m1.name AS title, m1.alias FROM #__menu AS m1 LEFT JOIN #__menu AS m2 ON m1.alias=m2.alias WHERE m1.id<>m2.id AND m1.type<>'menulink' AND m2.type<>'menulink' GROUP BY m1.id ORDER BY m1.alias";
        $db->setQuery($query);
        $duples = $db->loadObjectList();

        $url_prefix = 'index.php?option=com_menus&task=edit&cid[]=';

        if (count($duples)) {
            $list = array();
            $alias = $duples[0]->alias;
            foreach ($duples as $item) {
                if ($alias != $item->alias) {
                    $list[] = '';
                    $alias = $item->alias;
                }
                $list[] = '<a href="' . $url_prefix . $item->id . '">' . $item->title . '</a> [' . $item->menutype . ']';
            }
            $this->blob[] = array(
                'label' => MjHtml::label('', 'COM_MJ__WARNING_ALIASES'),
                'input' => MjHtml::text(implode('<br>', $list))
            );
        }
    }

    private function checkTemplateAssignments()
    {
        jimport('joomla.filesystem.file');
        jimport('joomla.filesystem.folder');
        $db = JFactory::getDbo();

        //get mobile templates
        $jpath_themes = JPATH_ROOT . '/templates';
        $templates = JFolder::folders($jpath_themes);
        $mobile_templates = array();
        foreach ($templates as $template) {
            if (is_file($jpath_themes . '/' . $template . '/templateDetails.xml')
                && is_file($jpath_themes . '/' . $template . '/index.php')
            ) {
                $content = file_get_contents($jpath_themes . '/' . $template . '/index.php');
                if (strpos($content, 'onGetMobileJoomla') !== false) {
                    $mobile_templates[] = $template;
                }
            }
        }

        if (count($mobile_templates) === 0) {
            return;
        }

        // get assigned mobile templates
        $list = array();
        foreach ($mobile_templates as $template) {
            $list[] = $db->quote($template);
        }
        $list = implode(', ', $list);

        $assigned_templates = array();
        if ($this->isJoomla15) {
            $query = "SELECT tm.template, tm.menuid, m.name FROM #__templates_menu AS tm LEFT JOIN #__menu AS m ON m.id=tm.menuid WHERE template IN ($list) AND tm.menuid>=0 AND tm.client_id=0 ORDER BY tm.template, tm.menuid";
            $db->setQuery($query);
            $rows = $db->loadObjectList();
            foreach ($rows as $row) {
                $assigned_templates[$row->template][] = array($row->menuid, $row->name);
            }
        } else {
            $joomlaWrapper = MjJoomlaWrapper::getInstance();
            $db = $joomlaWrapper->getDbo();

            $query = new MjQueryBuilder($db);
            $rows = $query
                ->select('template')
                ->from('#__template_styles')
                ->where("template IN ($list)")
                ->where('home=1')
                ->where('client_id=0')
                ->order('template')
                ->setQuery()
                ->loadObjectList();
            foreach ($rows as $row) {
                $assigned_templates[$row->template][] = array(0, null);
            }

            $query = new MjQueryBuilder($db);
            $rows = $query
                ->select('ts.template, m.id, m.title')
                ->from('#__menu AS m')
                ->leftJoin('#__template_styles AS ts ON m.template_style_id=ts.id')
                ->where("ts.template IN ($list)")
                ->where('ts.client_id=0')
                ->order('ts.template, m.id')
                ->setQuery()
                ->loadObjectList();
            foreach ($rows as $row) {
                $assigned_templates[$row->template][] = array($row->id, $row->title);
            }
        }

        if (count($assigned_templates) === 0) {
            return;
        }

        $url_prefix = 'index.php?option=com_menus&task=' . ($this->isJoomla15 ? 'edit&cid[]=' : 'item.edit&id=');

        $list = array();
        foreach ($assigned_templates as $key => $items) {
            foreach ($items as $item) {
                list($menuid, $title) = $item;
                if ($menuid) {
                    $list[] = $key . ' &lt; <a href="' . $url_prefix . $menuid . '">' . htmlspecialchars($title) . '</a>';
                } else {
                    $list[] = '<a href="index.php?option=com_templates">' . $key . '</a>'
                        . ' (' . JText::_('COM_MJ__WARNING_ASSIGNEDTEMPLATES_DEFAULT') . ')';
                }
            }
        }
        $this->blob[] = array(
            'label' => MjHtml::label('', 'COM_MJ__WARNING_ASSIGNEDTEMPLATES', 'COM_MJ__WARNING_ASSIGNEDTEMPLATES_DESC'),
            'input' => MjHtml::text(implode('<br>', $list))
        );
    }

    private function checkForcedMarkup()
    {
        $markup = isset($_COOKIE['mjmarkup']) ? $_COOKIE['mjmarkup'] : '';
        if ($markup === 'desktop' || $markup === '') {
            return;
        }

        $resetUrl = JUri::root() . '?device=desktop';
        $this->blob[] = array(
            'label' => MjHtml::label('', 'COM_MJ__WARNING_FORCEDMARKUP'),
            'input' => MjHtml::text(
                ucfirst($markup)
                . ' <a href="' . $resetUrl . '" target="_blank" class="btn btn-success btn-mini">'
                . JText::_('COM_MJ__WARNING_FORCEDMARKUP_RESET')
                . '</a>'
            )
        );
    }

    /**
     * @param MjSettingsModel $mjSettings
     */
    private function checkFavicon($mjSettings)
    {
        $modes = MjHelper::getDeviceList();
        $modes[''] = '';

        $mobile_templates = array();
        foreach ($modes as $mode => $dummy) {
            $template = $mjSettings->get($mode . '.template');
            if (!empty($template)) {
                $mobile_templates[$template] = 1;
            }
        }
        if (!$this->isJoomla15 && count($mobile_templates)) {
            $joomlaWrapper = MjJoomlaWrapper::getInstance();
            $db = $joomlaWrapper->getDbo();
            $query = new MjQueryBuilder($db);
            $templates = $query
                ->select('template')
                ->from('#__template_styles')
                ->where($query->qn('id') . ' IN (' . implode(', ', array_keys($mobile_templates)) . ')')
                ->where('client_id=0')
                ->setQuery()
                ->loadColumn();
            $mobile_templates = array();
            foreach ($templates as $template) {
                $mobile_templates[$template] = 1;
            }
        }
        foreach ($mobile_templates as $template => $dummy) {
            $favicon = JPATH_SITE . '/templates/' . $template . '/favicon.ico';
            if (is_file($favicon) && md5_file($favicon) === '4682009df58c0fe58ee462247355de56') {
                $this->blob[] = array(
                    'label' => MjHtml::label('', 'COM_MJ__WARNING_FAVICON'),
                    'input' => MjHtml::text(
                        JText::sprintf('COM_MJ__WARNING_FAVICON_TEXT', $template)
                    )
                );
            }
        }
    }

    private function checkPluginsOrdering()
    {
        $plugins = array(
            'mobilejoomla' => 'System - Mobile Core',
            'sef' => 'System - SEF',
            'mobilejoomlaopt' => 'System - Mobile Page Optimizer',
            'jch_optimize' => 'System - JCH Optimize',
            'cache' => 'System - Page Cache',
        );

        $orderings = array(
            array('mobilejoomla', 'mobilejoomlaopt'),
            array('mobilejoomla', 'cache'),
            array('sef', 'mobilejoomlaopt'),
            array('mobilejoomlaopt', 'jch_optimize'),
            array('mobilejoomlaopt', 'cache'),
            array('jch_optimize', 'cache'),
        );

        $joomlaWrapper = MjJoomlaWrapper::getInstance();
        $db = $joomlaWrapper->getDbo();
        $query = new MjQueryBuilder($db);
        $deps = array();
        foreach ($plugins as $name => $dummy) {
            $deps[] = $query->q($name);
        }
        $rows = $query
            ->select('element', 'ordering')
            ->from($this->isJoomla15 ? '#__plugins' : '#__extensions')
            ->where($query->qn('element') . ' IN (' . implode(', ', $deps) . ')')
            ->where($query->qn('folder') . '=' . $query->q('system'))
            ->setQuery()
            ->loadObjectList('element');

        foreach ($orderings as $ordering) {
            list($plugin1, $plugin2) = $ordering;
            if (JPluginHelper::isEnabled('system', $plugin1)
                && JPluginHelper::isEnabled('system', $plugin2)
                && $rows[$plugin1]->ordering >= $rows[$plugin2]->ordering
            ) {
                $this->blob[] = array(
                    'label' => MjHtml::label('', 'COM_MJ__WARNING_PLUGINS_ORDERING'),
                    'input' => MjHtml::text(
                        JText::sprintf('COM_MJ__WARNING_PLUGINS_ORDERING_TEXT', $plugins[$plugin1], $plugins[$plugin2])
                    )
                );
            }
        }
    }
}