<?php

/*
 * RESSIO Responsive Server Side Optimizer
 * https://github.com/ressio/
 *
 * @copyright   Copyright (C) 2013-2018 Kuneri, Ltd. All rights reserved.
 * @license     GNU General Public License version 2
 */

class Ressio_Cache_File implements IRessio_Cache
{
    /** @var IRessio_Filesystem */
    private $fs;
    /** @var IRessio_FileLock */
    private $filelock;
    /** @var string */
    private $cachedir;
    /** @var int */
    private $time;
    /** @var int */
    private $aging_time;
    /** @var array */
    private $id2file = array();

    /**
     * @param string $dir
     * @param int $ttl
     */
    public function __construct($dir = './cache', $ttl = null)
    {
        $this->cachedir = $dir;
        $this->time = time();
        $this->aging_time = $this->time - ($ttl === null ? 24 * 60 * 60 : $ttl);
    }

    /**
     * @param Ressio_DI $di
     * @throws ERessio_UnknownDiKey
     */
    public function setDI($di)
    {
        $this->fs = $di->filesystem;
        $this->filelock = $di->filelock;
        $config = $di->config;
        if (isset($config->cachedir)) {
            $this->cachedir = $config->cachedir;
        }
        if (isset($config->cachettl)) {
            $this->aging_time = $this->time - $config->cachettl;
        }
    }

    /**
     * @param string|array $deps
     * @param string $suffix
     * @return string
     */
    public function id($deps, $suffix = '')
    {
        if (is_array($deps)) {
            $deps = implode("\0", $deps);
        }
        return sha1($deps) . '_' . $suffix;
    }

    /**
     * @param string $id
     * @return string|bool
     */
    public function getOrLock($id)
    {
        $filename = $this->fileById($id);
        if ($this->fs->isFile($filename)) {
            if ($this->fs->getModificationTime($filename) <= $this->aging_time) {
                // update modification time to don't remove actively used cache file
                $this->fs->touch($filename);
            }
            return $this->fs->getContents($filename);
        }
        return $this->filelock->lock($filename);
    }

    /**
     * @param string $id
     * @param string $data
     * @return bool
     */
    public function storeAndUnlock($id, $data)
    {
        $filename = $this->fileById($id);
        if ($this->filelock->isLocked($filename, true)) {
            $this->fs->putContents($filename, $data);
            $this->filelock->unlock($filename);
            return true;
        }
        return false;
    }

    /**
     * @param string $id
     * @return bool
     */
    public function delete($id)
    {
        $filename = $this->fileById($id);
        if (!$this->fs->isFile($filename)) {
            return true;
        }
        if (!$this->filelock->lock($filename)) {
            return false;
        }
        $this->fs->delete($filename);
        $this->filelock->unlock($filename);
        return true;
    }

    /**
     * @param string $id
     * @return string
     */
    private function fileById($id)
    {
        if (!isset($this->id2file[$id])) {
            $dir = $this->cachedir . '/' . substr($id, 0, 2);
            $this->fs->makeDir($dir);
            $this->id2file[$id] = $dir . '/' . $id;
        }
        return $this->id2file[$id];
    }
}

